<?php

error_reporting(E_ALL);
ini_set('display_errors', 1);
header("Content-Type: application/json");

require '../dbconn.php';
require_once "../../middlewares/verifyJWT.php";

/*
// CHECK AUTH
$auth = verifyJWT();
if (!$auth["success"]) {
    echo json_encode(["success" => false, "message" => "Unauthorized"]);
    exit;
}
*/

// ALLOW ONLY GET
if ($_SERVER["REQUEST_METHOD"] !== "GET") {
    echo json_encode(["success" => false, "message" => "Invalid request method. Use GET"]);
    exit;
}

// GET PARAMETERS
$page = isset($_GET["page"]) ? (int)$_GET["page"] : 1;
$limit = isset($_GET["limit"]) ? (int)$_GET["limit"] : 10;
$search = isset($_GET["search"]) ? trim($_GET["search"]) : "";
$category_ref_id = isset($_GET["category_ref_id"]) ? trim($_GET["category_ref_id"]) : "";

// ❌ category_ref_id missing
if ($category_ref_id === "" || !is_numeric($category_ref_id)) {
    echo json_encode([
        "success" => false,
        "message" => "category_ref_id is required"
    ]);
    exit;
}

// PAGINATION VALIDATION
if ($page < 1) $page = 1;
if ($limit < 1 || $limit > 100) $limit = 10;
$offset = ($page - 1) * $limit;

// WHERE CLAUSE
$where_clause = "WHERE p.active = 1 AND p.category_ref_id = ?";
$params = [$category_ref_id];
$types = "i";

// SEARCH
if (!empty($search)) {
    $where_clause .= " AND p.name LIKE ?";
    $params[] = "%" . $search . "%";
    $types .= "s";
}

// COUNT TOTAL
$sql_count = "SELECT COUNT(*) as total FROM products p $where_clause";
$stmt_count = $conn->prepare($sql_count);
$stmt_count->bind_param($types, ...$params);
$stmt_count->execute();

$total_result = $stmt_count->get_result();
$total_row = $total_result->fetch_assoc();
$total_records = $total_row["total"];
$total_pages = ceil($total_records / $limit);

// FETCH PRODUCTS + VARIANTS
$sql = "
SELECT 
    p.product_id,
    p.name,
    p.description,
    p.image_url,
    p.category_ref_id,
    p.scientific_name,
    p.base_price,
    c.name AS category_name,

    COALESCE(
        JSON_ARRAYAGG(
            JSON_OBJECT(
                'product_variant_id', pv.product_variant_id,
                'product_ref_id', pv.product_ref_id,
                'sku', pv.sku,
                'price', pv.price,
                'stock_quantity', pv.stock_quantity,
                'cut_and_clean_price', pv.cut_and_clean_price,
                'image_url',
                    CASE
                        WHEN pv.image_url IS NULL OR pv.image_url = ''
                        THEN p.image_url
                        ELSE pv.image_url
                    END,
                'specifications', pv.specifications,
                'is_default', pv.is_default,
                'in_stock', pv.in_stock,
                'name', pv.name,
                'description', pv.description,
                'created_at', pv.created_at,
                'updated_at', pv.updated_at
            )
        ),
        JSON_ARRAY()
    ) AS variants

FROM products p
LEFT JOIN categories c 
    ON p.category_ref_id = c.category_id
LEFT JOIN product_variants pv 
    ON pv.product_ref_id = p.product_id

$where_clause
GROUP BY p.product_id
ORDER BY p.created_at DESC
LIMIT ? OFFSET ?
";

$stmt = $conn->prepare($sql);

// ADD PAGINATION PARAMS
$params_for_query = $params;
$params_for_query[] = $limit;
$params_for_query[] = $offset;
$types_for_query = $types . "ii";

$stmt->bind_param($types_for_query, ...$params_for_query);
$stmt->execute();

$result = $stmt->get_result();

$products = [];
while ($row = $result->fetch_assoc()) {

    // ✅ Decode variants JSON
    $variants = json_decode($row['variants'], true);
    if (!is_array($variants)) {
        $variants = [];
    }

    // ✅ Decode specifications for each variant
    foreach ($variants as &$variant) {

    if (isset($variant['specifications'])) {

        // Decode ONLY if it's a JSON string
        if (is_string($variant['specifications'])) {
            $decoded = json_decode($variant['specifications'], true);
            $variant['specifications'] = is_array($decoded) ? $decoded : [];
        }

        // If null
        if ($variant['specifications'] === null) {
            $variant['specifications'] = [];
        }
    }
}


    $row['variants'] = $variants;
    $products[] = $row;
}

// FINAL RESPONSE
echo json_encode([
    "success" => true,
    "data" => $products,
    "pagination" => [
        "current_page" => $page,
        "total_pages" => $total_pages,
        "total_records" => $total_records,
        "limit" => $limit,
        "has_next" => $page < $total_pages,
        "has_prev" => $page > 1
    ],
    "search" => $search,
    "category_ref_id" => $category_ref_id
]);

$stmt->close();
$stmt_count->close();
$conn->close();


?>
