<?php
header("Content-Type: application/json");
require '../dbconn.php'; // Database connection

// Check if request method is POST
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $code = isset($_POST['code']) ? trim($_POST['code']) : '';
    $cart_total = isset($_POST['cart_total']) ? floatval($_POST['cart_total']) : 0;
    
    if ($code === '' || $cart_total <= 0) {
        echo json_encode(["success" => false, "message" => "Invalid input"]);
        exit;
    }
    
    // Check if promo code exists and is active
    $stmt = $conn->prepare("SELECT id, type, discount_value, minimum_purchase, maximum_discount, usage_limit, times_used, is_active, valid_from, valid_until FROM promocodes WHERE code = ? AND is_active = 1");
    $stmt->bind_param("s", $code);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows === 0) {
        echo json_encode(["success" => false, "message" => "Invalid or expired promo code"]);
        exit;
    }
    
    $promo = $result->fetch_assoc();
    $current_date = date("Y-m-d H:i:s");
    
    // Validate date range
    if ($current_date < $promo['valid_from'] || $current_date > $promo['valid_until']) {
        echo json_encode(["success" => false, "message" => "Promo code is not valid at this time"]);
        exit;
    }
    
    // Check minimum purchase amount
    if ($cart_total < $promo['minimum_purchase']) {
        echo json_encode(["success" => false, "message" => "Cart total is below the minimum purchase requirement"]);
        exit;
    }
    
    // Check usage limit
    if ($promo['times_used'] >= $promo['usage_limit']) {
        echo json_encode(["success" => false, "message" => "Promo code usage limit reached"]);
        exit;
    }
    
    // Calculate discount amount
    $discount_amount = 0;
    if ($promo['type'] === 'percentage') {
        $discount_amount = ($cart_total * $promo['discount_value']) / 100;
        if ($discount_amount > $promo['maximum_discount']) {
            $discount_amount = $promo['maximum_discount'];
        }
    } elseif ($promo['type'] === 'FIXED') {
        $discount_amount = min($promo['discount_value'], $promo['maximum_discount']);
        
    }
    
    echo json_encode([
        "success" => true,
        "data" => [
            "valid" => true,
            "discount_type" => $promo['type'],
            "discount_value" => $promo['discount_value'],
            "discount_amount" => $discount_amount,
            "minimum_purchase" => $promo['minimum_purchase'],
            "maximum_discount" => $promo['maximum_discount']
        ]
    ]);
} else {
    echo json_encode(["success" => false, "message" => "Invalid request method"]);
}
?>
